# Before `make install' is performed this script should be runnable with
# `make test'. After `make install' it should work as `perl Data-Traverse.t'

use strict;

#########################

use Test::More qw(no_plan);
BEGIN { use_ok('Data::Traverse') };
use Data::Traverse qw( :all);

#########################

my $data= { h1_01_key1 => { h2_01_key1 => [ qw(a3_01_item1 a3_01_item2 a3_01_item3) ],
                            h2_01_key2 => { h3_01_key1 => 'h3_01_val1', h3_01_key2 => 'h3_01_val2',  },
                            h2_01_key3 => 'h2_01_val1',
                          },
            h1_01_key2 => [ { h3_02_key1 => 'h3_02_val1', h3_02_key2 => [ qw(a4_01_item1 a4_01_item2)] }, 
                            { h3_03_key1 => 'h3_03_val1', h3_03_key2 => [ qw(a4_02_item1 a4_02_item2)] }, 
                            [ 'a3_02_item1', { h4_01_key1 => 'h4_01_val1', h4_01_key2 => 'h4_01_val2'} ],
                          ],
          };

my @scalars_level= ('', '', 'h2_01_val1',
                    'a3_01_item1 a3_01_item2 a3_01_item3 h3_01_val1 h3_01_val2 h3_02_val1'
                      . ' h3_03_val1 a3_02_item1' ,
                    'a4_01_item1 a4_01_item2 a4_02_item1 a4_02_item2 h4_01_val1 h4_01_val2',
                    '',
                   );

my @hash_keys=( 'h1_01_key1 : h1_01_key2', 'h2_01_key1 : h2_01_key2 : h2_01_key3',
                'h3_01_key1 : h3_01_key2 - h3_02_key1 : h3_02_key2 - h3_03_key1 : h3_03_key2',
                'h4_01_key1 : h4_01_key2',
                '',
              );

foreach my $level (0..5)
  { is( list( scalars_level( $data, $level)), $scalars_level[$level] , "scalars_level($level)"); 
    my $list= scalars_level( $data, $level);
    is( list( @$list), $scalars_level[$level] , "scalars_level($level) (through scalar)"); 
  }

is( list( scalars_level( $data, 8)), "", "scalars_level(8)");

is( list( scalars( $data)), join( " ", grep { $_ } @scalars_level), "scalars");
my $list= scalars( $data);
is( list( @$list), join( " ", grep { $_ } @scalars_level), "scalars (through scalar)");

foreach my $level (0..5)
  { is( list( scalars( $data, $level)), 
        join( " ", grep { $_ } @scalars_level[0..$level]),
        "scalars( up to level $level)"
      );
  }

foreach my $level (0..4)
  { is( hash_keys( refs_level( $data, HASH => $level)), $hash_keys[$level] , "hashes($level)"); 
    my $refs_level= refs_level( $data, HASH => $level);
    is( hash_keys( @$refs_level), $hash_keys[$level] , "hashes($level) (through scalar)"); 
  }

is( hash_keys( refs( $data, 'HASH')), join( " - ", grep { $_ } @hash_keys), "hashes");
my $refs= refs( $data, 'HASH');
is( hash_keys( @$refs), join( " - ", grep { $_ } @hash_keys), "hashes (through scalar)");


foreach my $level (0..4)
  { is( hash_keys( refs( $data, HASH => $level)), 
        join( " - ", grep { $_ } @hash_keys[0..$level]),
        "hashes( up to level $level)"
      );
  }

my $result='';
my $t= Data::Traverse->new( $data);
$t->traverse( sub { $result.= "$_:" if( !ref( $_) && m{^a.*item1$}) } );
is( $result, 'a3_01_item1:a4_01_item1:a4_02_item1:a3_02_item1:', "traverse");

$result='';
$t->traverse( sub { $result.= "$_:" if( $t->path eq '/h1_01_key1/h2_01_key1/1'); });
is( $result, 'a3_01_item2:', 'path (including array)');

$result='';
$t->traverse( sub { $result.= "$_:" if( $t->path eq '/h1_01_key1/h2_01_key3'); });
is( $result, 'h2_01_val1:', 'path');

$result='';
$t->traverse( sub { $result.= "$_:" if( $t->path_matches( qr{/h3_01_key\d$})); });
is( $result, 'h3_01_val1:h3_01_val2:', 'path_matches');

$result='';
$data->{o1}= bless [ qw/foo bar/ ], 'obj';
$t->traverse( sub { $result.= "$_:" if( ref( $t->parent) eq 'obj'); });
is( $result, 'foo:bar:', 'parent');

$result='';
$t->traverse( sub { my $arrays=  grep { ref $_ eq 'ARRAY' } $t->ancestors;
                   $result .= "$_:" if( $t->is_scalar && ($arrays == 2)); 
                  }
            );
is( $result, 'a4_01_item1:a4_01_item2:a4_02_item1:a4_02_item2:a3_02_item1:h4_01_val1:h4_01_val2:', 'ancestors (and is_scalar)');

$result='';
$t->traverse( sub { $result.= "$_:" if( $t->is_scalar && $t->item_index == 2); });
is( $result, 'a3_01_item3:', 'item_index');

$result='';
$t->traverse( sub { $result.= "$_:" if( $t->item_key eq 'h2_01_key3'); });
is( $result, 'h2_01_val1:', 'item_key');

$result='';
$t->traverse( sub { $result.= "$_:" if( $t->is_scalar && $t->parent_index == 2); });
is( $result, 'a3_02_item1:', 'parent_index');

$result='';
$t->traverse( sub { $result.= "$_:" if( $t->parent_key eq 'h2_01_key2'  && $t->is_scalar ); });
is( $result, 'h3_01_val1:h3_01_val2:', 'parent_key');

$result='';
$t->traverse( sub { if( $t->is_scalar && $t->parent_key eq 'h2_01_key2'){ $result .=$_; $t->finish; } });
is( $result, 'h3_01_val1', 'prune');

$result='';
$t->traverse( sub { if( $t->parent_index == 1 && $t->is_scalar ){ $result .=$_; $t->finish; } });
is( $result, 'h3_03_val1', 'prune');

$result='';
$t->traverse( sub { $result.= "$_:" if( $t->is_scalar && $t->level == 2); });
is( $result, 'h2_01_val1:foo:bar:', 'level');

$result='';
$t->traverse( sub { $result .= "$_:" if( $t->is_scalar); $t->prune if( $t->level > 1); });
is( $result, 'h2_01_val1:foo:bar:','prune');

$result='';
my $data_with_cycle= { k1 => 'v1', k2 => 'v2' };
$t= Data::Traverse->new( $data_with_cycle);
$t->traverse( sub { $result.= $_ if( $t->item_key); });
is( $result, 'v1v2', 'data without cycle');

$result='';
$data_with_cycle->{k3}= $data_with_cycle;
$t->traverse( sub { $result.= $_ if( $t->item_key); });
is( $result, 'v1v2', 'data with cycle');

$result='';
my $data_with_undef= { k1 => undef, k2 => 'v2' };
$t= Data::Traverse->new( $data_with_undef);
$t->traverse( sub { $result.= $_ || '' if( $t->item_key); });
is( $result, 'v2', 'data with undef');


# utility functions used to compare actual results to expected results
sub list { return join " ", @_; }
sub hash_keys { return join( ' - ', map { join " : ", sort keys %$_ } @_); }

